"""
persistence.py:
contains what we need to make a (category) objects (see models.py) persist
beyond application scope and load them from outside sources

For now we can save, load and list
* to/from a file (using a turtle rdf graph serialization for categories)
* to/from a github repository (using a turtle rdf graph serialization
for categories)
"""
from abc import ABCMeta, abstractmethod
from catedit import github, app
from base64 import b64decode
from flask.ext.github import GitHubError
import os
import json

logger = app.logger


class Persistence:
    """
        Meta-class for all persistence classes
    """
    __metaclass__ = ABCMeta

    @abstractmethod
    def session_compliant(self):
        """
            Abstract - Can this persistence submit a changeset?
        """
        return

    @abstractmethod
    def submit_changes(self, **kwargs):
        """
            Abstract - Submit all saved objects, only useful if persistence
            method can submit a changeset
        """
        return

    @abstractmethod
    def save(self, **kwargs):
        """
            Abstract - Saves object
        """
        return

    @abstractmethod
    def delete(self, **kwargs):
        """
            Abstract - Deletes object
        """
        return

    @abstractmethod
    def load(self, **kwargs):
        """
            Abstract - Loads object
        """
        return

    @abstractmethod
    def list(self, **kwargs):
        """
            Abstract - Lists objects
        """
        return


class PersistenceToFile(Persistence):
    """
        Persistence Class for saving to a local file (see config.py
        for tweaking)

        Expected kwargs for saving to/loading from a file are:
        * name : name of the file to write in/read from
        * content : desired content of the file when writing
    """
    @property
    def session_compliant(self):
        """
            Not session compliant: each modification is submitted
        """
        return False

    def submit_changes(self, **kwargs):
        """
            As each modification is submitted, this is where we save to a file
        """

    def save(self, **kwargs):
        """
            Saves to a file
        """
        path_to_save = app.config["FILE_SAVE_DIRECTORY"]+kwargs["name"]
        file_to_save = open(path_to_save, 'wb')
        file_to_save.write(kwargs["content"])
        file_to_save.close()

    def load(self, **kwargs):
        """
            Loads from a file
        """
        path_to_load = app.config["FILE_SAVE_DIRECTORY"]+kwargs["name"]
        file_to_load = open(path_to_load, 'rb')
        file_content = file_to_load.read()
        file_to_load.close()
        return file_content

    def delete(self, **kwargs):
        """
            Deletes a file
        """
        path_to_delete = app.config["FILE_SAVE_DIRECTORY"]+kwargs["name"]
        os.remove(path_to_delete)

    # IDEA: return { file_name: file_content } type dict
    def list(self, **kwargs):
        """
            Lists all files in file directory (as set in config.py)
        """
        file_content_list = []
        for file_name in os.listdir(app.config["FILE_SAVE_DIRECTORY"]):
            if not file_name or file_name[0] == ".":
                continue
            path_to_load = open(app.config["FILE_SAVE_DIRECTORY"]+file_name)
            file_content = path_to_load.read()
            path_to_load.close()
            file_content_list.append(file_content)
        # logger.debug(file_content_list)
        return file_content_list


class PersistenceToGithub(Persistence):
    """
        Persistence Class for saving to/loading from a Github repository (see
        config.py for tweaks)

        Expected kwargs for saving to/loading from a Github repository are:
        * name : name of the file to write in/read from
        * content : desired content of the file when writing
        * message : when saving or deleting, commit message to be saved
        to Github
    """
    def __init__(self, **kwargs):
        self.repository = kwargs.get("repository", "")

    @property
    def session_compliant(self):
        """
            Not session compliant: each modification is comitted
        """
        return True

    def save(self, **kwargs):
        """
            Saves all the recorded files in the session dict to a Github
            repository

            Expected kwargs is:
            * message: the commit message to document the changes
            * deletion_list : the list of files to delete, list of dicts, each
            one of the format:
            {"name": name_of_the_file}
            * modification_list: the list of the files to change, list of
            dicts, each one of the format:
            {"name": name_of_the_file, "content": content_of_the_file}

            IMPORTANT: To save to a file Github, we have to use Git
            internal mechanics:
            1) Get the current reference for master branch
            2) Get the latest commit on that reference
            3) Get the tree associated to this commit
            4) Create a new tree using the old tree and the session dict that
            contains all the changes (keys are "modified_categories" and
            "deleted_categories")
                4-1) This requires creating new blobs for new files
            5) Create a new commit referencing the previous commit as parent
            and the new tree we just created
            6) Update the master branch reference so it points on this new
            commit

            About point 4):
            We have 3 list of elements: files already in repo, modified files,
            and deleted_files
            First, we loop on the files already in repo: this allows us to
            update it (if it's in both original file list and modified files
            list) or delete it (if it's in both original file list and deleted
            files list)
            Then, we loop on the modified files list. If it isn't in the
            original file list, then it's a new file and we append it to the
            tree
        """

        deletion_list = kwargs["deletion_list"]
        modification_list = kwargs["modification_list"]

        # point 1
        try:
            ref_master = github.get(
                "repos/"
                + app.config["REPOSITORY_OWNER"] + "/"
                + self.repository
                + "/git/refs/heads/master"
            )
            logger.debug(str(ref_master))
        except GitHubError as ghe:
            logger.debug("GitHubError trying to get the reference \
                         to the master branch")
            logger.debug(
                "Endpoint: "
                + "repos/"
                + app.config["REPOSITORY_OWNER"] + "/"
                + self.repository
                + "/git/refs/heads/master"
            )
            logger.debug(ghe.response.text)
        logger.debug(str(github.get("rate_limit")["resources"]))

        # point 2
        try:
            last_commit_master = github.get(
                "repos/"
                + app.config["REPOSITORY_OWNER"] + "/"
                + self.repository
                + "/git/commits/"
                + ref_master["object"]["sha"]
            )
            logger.debug(str(last_commit_master))
        except GitHubError as ghe:
            logger.debug("GitHubError trying to get the commit associated \
                         to the latest reference to the master branch")
            logger.debug(ghe.response.text)
        logger.debug(str(github.get("rate_limit")["resources"]))

        # Point 3
        try:
            last_commit_tree = github.get(
                "repos/"
                + app.config["REPOSITORY_OWNER"] + "/"
                + self.repository
                + "/git/trees/"
                + last_commit_master["tree"]["sha"]
                + "?recursive=1"
            )
            logger.debug(str(last_commit_tree))
        except GitHubError as ghe:
            logger.debug("GitHubError trying to get the tree from the commit \
                         associated to the latest reference to the master \
                         branch")
            logger.debug(ghe.response.text)
        logger.debug(str(github.get("rate_limit")["resources"]))

        # Point 4
        new_tree_data = {"tree": []}

        # First we loop over the existing elements to spot which are modified
        # and which are untouched and create new blobs when needed
        for element in last_commit_tree["tree"]:
            logger.debug(element)
            if element["type"] == "blob":
                # test if element is in deleted categories, if it is,
                # no point doing anything, the file won't be in the new tree
                if not(
                    element["path"] in [
                        (app.config["CATEGORIES_PATH"] + category["name"])
                        for category in deletion_list
                    ]
                ):

                    # the element is either modified or untouched so in both
                    # case we'll need a blob sha
                    blob_sha = ""

                    # test if element is in modified categories
                    if (
                        element["path"] in [
                            (app.config["CATEGORIES_PATH"] + category["name"])
                            for category in modification_list
                        ]
                    ):
                        # find element in modified categories
                        for category in modification_list:
                            if element["path"] == (
                                app.config["CATEGORIES_PATH"] + category["name"]
                            ):
                                # 4-1 for modified files, creating a new blob
                                new_blob_data = {
                                    "content": category["content"],
                                    "encoding": "utf-8"
                                }
                                try:
                                    new_blob = github.post(
                                        "repos/"
                                        + app.config["REPOSITORY_OWNER"] + "/"
                                        + self.repository
                                        + "/git/blobs",
                                        data=new_blob_data
                                    )
                                    blob_sha = new_blob["sha"]
                                    break
                                except GitHubError as ghe:
                                    logger.debug(
                                        "GitHubError trying to post a new \
                                        blob with following data: "
                                        + str(new_blob_data)
                                    )
                                    logger.debug(ghe.response.text)
                                logger.debug(str(github.get("rate_limit")["resources"]))

                    # this means element is an untouched file so we just get
                    # its sha
                    else:
                        blob_sha = element["sha"]
                    new_tree_data["tree"].append({"path": element["path"],
                                                  "mode": element["mode"],
                                                  "type": "blob",
                                                  "sha": blob_sha})
        logger.debug(str(new_tree_data["tree"]))
        # Now we loop over modified categories to find the ones that don't
        # exist yet in the last commit tree in order to create blobs for them
        for category in modification_list:
            logger.debug(app.config["CATEGORIES_PATH"]+category["name"]
                         + " should not be in "
                         + str([elt["path"] for
                                elt in last_commit_tree["tree"]]))
            if (app.config["CATEGORIES_PATH"] + category["name"] not in
                    [file["path"] for file in last_commit_tree["tree"]]):

                # 4-1 for added files, creating a new blob
                new_blob_data = {"content": category["content"],
                                 "encoding": "utf-8"}
                try:
                    new_blob = github.post(
                        "repos/"
                        + app.config["REPOSITORY_OWNER"] + "/"
                        + self.repository
                        + "/git/blobs",
                        data=new_blob_data
                    )
                except GitHubError as ghe:
                    logger.debug(
                        "GitHubError trying to post a new blob with following \
                        data: "
                        + str(new_blob_data)
                    )
                    logger.debug(ghe.response.text)
                logger.debug(str(github.get("rate_limit")["resources"]))
                new_tree_data["tree"].append({
                    "path": app.config["CATEGORIES_PATH"] + category["name"],
                    "mode": "100644",
                    "type": "blob",
                    "sha": new_blob["sha"]
                })
        logger.debug(str(new_tree_data))

        # Finally, we post the new tree
        try:
            new_tree_response = github.post(
                "repos/"
                + app.config["REPOSITORY_OWNER"]+"/"
                + self.repository
                + "/git/trees",
                data=new_tree_data
            )
        except GitHubError as ghe:
            logger.debug(
                "GitHubError trying to post a new tree with following data: "
                + str(new_tree_data)
            )
            logger.debug(ghe.response.text)

        # Point 5
        new_commit_data = {"message": kwargs["message"],
                           "parents": [last_commit_master["sha"]],
                           "tree": new_tree_response["sha"]}
        logger.debug(str(new_commit_data))
        try:
            new_commit = github.post(
                "repos/"
                + app.config["REPOSITORY_OWNER"]+"/"
                + self.repository
                + "/git/commits",
                data=new_commit_data
            )
            logger.debug(str(new_commit))
        except GitHubError as ghe:
            logger.debug(
                "GitHubError trying to post a new commit with following data: "
                + str(new_commit_data)
            )
            logger.debug(ghe.response.text)
        logger.debug(str(github.get("rate_limit")["resources"]))

        # Point 6
        new_head_data = {"sha": new_commit["sha"], "force": "true"}
        logger.debug(str(new_head_data))
        try:
            new_head = github.patch(
                "repos/"
                + app.config["REPOSITORY_OWNER"] + "/"
                + self.repository
                + "/git/refs/heads/master",
                data=json.dumps(new_head_data)
            )
            logger.debug(str(new_head))
        except GitHubError as ghe:
            logger.debug(
                "GitHubError trying to edit the head to the master branch \
                with the following data: "
                + str(new_head_data)
            )
            logger.debug(ghe.response.text)
        logger.debug(str(github.get("rate_limit")["resources"]))


    def load(self, **kwargs):
        """
            Loads from a Github repository
        """
        try:
            filedict = github.get("repos/"
                                  + app.config["REPOSITORY_OWNER"]+"/"
                                  + self.repository
                                  + "/contents/"
                                  + app.config["CATEGORIES_PATH"]
                                  + kwargs["name"])
            file_content = str(b64decode(filedict["content"]), "utf-8")
        except GitHubError as ghe:
            logger.debug("Github Error trying to get file: "+kwargs["name"])
            logger.debug("Github sent an error, if 404, either you may not \
                         have access to the repository or it doesn't exist ")
            logger.debug(ghe.response.text)
        logger.debug(str(github.get("rate_limit")["resources"]))
        return file_content

    def delete(self, **kwargs):
        """
            Deletes from a Github repository

            Expected kwargs are:
            * name : the name of the file to delete
            * message : the commit message for the deletion

        """
        pass

    def list(self, **kwargs):
        """
            Lists all files in the github repository (as set in config.py)
        """
        filenames_list = []
        try:
            files_in_repo = github.get("repos/"
                                       + app.config["REPOSITORY_OWNER"]+"/"
                                       + self.repository
                                       + "/contents/"
                                       + app.config["CATEGORIES_PATH"])
            filenames_list = [github_file["name"]
                              for github_file in files_in_repo]
            # logger.debug(filenames_list)
        except GitHubError as ghe:
            logger.debug("Github Error trying to get the file list in the \
                         category repository")
            logger.debug("NOTE: Github sent an error, if 404 either you \
                         may not have access to the repository or it doesn't \
                         exist or there isn't any files in it")
            logger.debug(ghe.response.text)
        logger.debug(str(github.get("rate_limit")["resources"]))

        file_content_list = []
        for filename in filenames_list:
            try:
                filedict = github.get("repos/"
                                      + app.config["REPOSITORY_OWNER"]+"/"
                                      + self.repository
                                      + "/contents/"
                                      + app.config["CATEGORIES_PATH"]
                                      + filename)
                file_content_list.append(str(b64decode(filedict["content"]),
                                         "utf-8"))
            except GitHubError as ghe:
                logger.debug("Github Error trying to get file: "+filename)
                logger.debug(ghe.response.text)
        logger.debug(str(github.get("rate_limit")["resources"]))
        # logger.debug(file_content_list)
        return file_content_list
