"""
__init__.py:
module main file used to configure the Flask app
"""

from logging import FileHandler, Formatter
import os
import re
from catedit.version import CURRENT_VERSION
from requests import request

from flask import has_request_context, Flask, session, request, url_for, g
from flask_wtf.csrf import CsrfProtect
from flask.ext.github import GitHub
from flask.ext.cache import Cache
from flask.ext.restful import Api
from celery import Celery

from catedit.settings import AppSettings

# Set up app
app = Flask(__name__)
app.config.from_object(AppSettings)
app_configured = False
try:
    from catedit.config import AppConfig
    app.config.from_object(AppConfig)
    app_configured = True
except ImportError:
    pass

if 'CATEDIT_SETTINGS' in os.environ:
    app.config.from_envvar('CATEDIT_SETTINGS')
    app_configured = True

if not app_configured:
    raise Exception("Catedit not configured")

cache = Cache(app, config=app.config["CACHE_CONFIG"])

# CSRF protection
csrf = CsrfProtect(app)

# Github
app.config["GITHUB_CLIENT_ID"] = app.config["PERSISTENCE_CONFIG"].get(
    "GITHUB_CLIENT_ID",
    "local_persistence"
)
app.config["GITHUB_CLIENT_SECRET"] = app.config["PERSISTENCE_CONFIG"].get(
    "GITHUB_CLIENT_SECRET",
    "local_persistence"
)
github = GitHub(app)

#celery = Celery(app.name, broker=app.config['CELERY_BROKER_URL'], backend=app.config['CE'])
celery = Celery(app.name, broker=app.config['CELERY_BROKER_URL'], backend=app.config['CELERY_RESULT_BACKEND'])
celery.conf.update(app.config)

@github.access_token_getter
def token_getter():
    """
        Utility function for github-flask module to get user token when
        making authenticated requests
    """
    if (has_request_context() and session.get("user_logged", None)):
        # logger.debug("I made an authentified request")
        return session["user_code"]
    else:
        return g.get("user_code")

def log_api_rate(r, *args, **kwargs):
    """
        Utility hook function to link to every github call as a kwarg so the
        app logs how many requests can still be made, after the current request
    """
    app.logger.debug(
        str(r.request.method) + " "
        + str(r.url) + " - "
        + "Remaining requests count: "
        + str(r.headers["X-RateLimit-Remaining"]) + "/"
        + str(r.headers["X-RateLimit-Limit"])
    )

def save_links(r, *args, **kwargs):
    """
        Utility hook function that stores the links in the header of
        the response in order to use them for further API requests.

        After that hook, the entry "pagination_links" in session is updated
        with last page and current_page, as well as the resource to request
        from for each header link
    """
    session["pagination_links"] = {}
    log_api_rate(r, *args, **kwargs)
    if r.headers.get("link", None) is not None:
        session["pagination_links"] = r.links
        for (key, item) in session["pagination_links"].items():
            resource = item["url"][len(github.BASE_URL):]
            item["url"] = resource
        if session["pagination_links"].get("next", None) is not None:
            page_arg = re.search(
                "(\?|&)page=(\d+)",
                string=session["pagination_links"]["next"]["url"]
            )
            session["pagination_links"]["current_page"] = int(
                page_arg.group(2)
            )-1
            if session["pagination_links"].get("last", None) is not None:
                last_page_arg = re.search(
                    "(\?|&)page=(\d+)",
                    string=session["pagination_links"]["last"]["url"]
                )
                session["pagination_links"]["last_page"] = int(
                    last_page_arg.group(2)
                )
            else:
                # We don't know what is the last page (case: github commits
                # API)
                session["pagination_links"]["last_page"] = \
                session["pagination_links"]["current_page"] + 1
        elif session["pagination_links"].get("prev", None) is not None:
            # This means we're at the last page
            page_arg = re.search(
                "(\?|&)page=(\d+)",
                string=session["pagination_links"]["prev"]["url"]
            )
            session["pagination_links"]["current_page"] = int(
                page_arg.group(2)
            )+1
            session["pagination_links"] \
                   ["last_page"] = session["pagination_links"]["current_page"]
        else:
            session["pagination_links"]["current_page"] = 1
            session["pagination_links"]["last_page"] = 1
    else:
        session.pop("pagination_links", None)

# Api
api = Api(app)

# Version
app.config["CURRENT_VERSION"] = CURRENT_VERSION

# Views and APIs
from catedit.views import home, categories, social, meta
from catedit.resources import CategoryAPI, CategoryChangesAPI
app.register_blueprint(home.module)
app.register_blueprint(categories.module, url_prefix='/cat')
app.register_blueprint(social.module, url_prefix='/social')
app.register_blueprint(meta.meta, url_prefix='/meta')


api.add_resource(CategoryAPI,
                 '/category/<string:cat_id>',
                 '/category',
                 endpoint='category')
api.add_resource(CategoryChangesAPI,
                 '/category-changes/<string:cat_id>',
                 '/category-changes',
                 endpoint='category_changes')

# Pagination utility functions for templates
def url_for_other_page(page, arg_name="page"):
    args = request.view_args.copy()
    args[arg_name] = page
    return url_for(request.endpoint, **args)
app.jinja_env.globals['url_for_other_page'] = url_for_other_page

def url_for_other_per_page(
    per_page,
    page_arg_name="page",
    per_page_arg_name="per_page"
):
    args = request.view_args.copy()
    args[page_arg_name] = 1
    args[per_page_arg_name] = per_page
    return url_for(request.endpoint, **args)
app.jinja_env.globals['url_for_other_per_page'] = url_for_other_per_page

# Set up logging
if app.config["LOGGING_CONFIG"]["IS_LOGGING"]:
    file_handler = FileHandler(filename=app.config["LOGGING_CONFIG"]
                                                  ["LOG_FILE_PATH"])
    file_handler.setFormatter(Formatter(
        '%(asctime)s %(levelname)s: %(message)s '
        '[in %(pathname)s:%(lineno)d]',
        '%Y-%m-%d %H:%M:%S'
    ))
    app.logger.addHandler(file_handler)
    app.logger.setLevel(app.config["LOGGING_CONFIG"]["LOGGING_LEVEL"])

# Session management

app.secret_key = app.config["SECRET_KEY"]
