"""
home.py:
The views functions that handle authentication and index pages
"""
import json
from catedit import app, github, log_api_rate
from flask import render_template, redirect, url_for, \
                  session, Blueprint
from flask.ext.github import GitHubError
from base64 import b64decode

module = Blueprint('home', __name__)
logger = app.logger


@module.route('/', methods=['GET'])
@module.route('/index', methods=['GET'])
def index():
    """
        View for index page (for now it's only a readme text so no computing
        is required)
    """
    return render_template("home/index.html")

@module.route('/catedit-login', methods=["GET"])
def login():
    """
        Function called if the user is new or revoked the auth token
    """
    if not session.get("user_logged", False):
        return github.authorize(
            scope="repo",
            redirect_uri=url_for('home.login_callback', _external=True)
        )
    else:
        return redirect(url_for('home.index'))


@module.route('/catedit-github-callback')
@github.authorized_handler
def login_callback(oauth_code):
    """
        Function that handles callback from Github after succesful login
    """
    session.permanent = False
    if session.get("user_code", None) is None:
        # That means we got here using github callback
        # and not the login form
        session["user_code"] = oauth_code
    logger.debug(session["user_code"])
    session["user_logged"] = True
    session["user_login"] = "auth-error"
    
    session["properties"] = {
        repo: {} for repo
        in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]
    }
    session["modified_categories"] = {
        repo: {} for
        repo in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]
    }
    session["deleted_categories"] = {
        repo: {} for repo
        in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]
    }
    
    try:
        session["user_login"] = github.get(
            "user",
            hooks=dict(response=log_api_rate)
        )["login"]
    except GitHubError as ghe:
        logger.error(
            "GitHubError trying to get the user login"
        )
        logger.error(ghe.response.text)
    try:
        repo_list = []
        repo_list = github.get("user/repos")

        user_repos_name = [repo["name"] for repo in repo_list]
        session["user_repositories"] = list(
            set(user_repos_name).intersection(
                app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]
            )
        )
        session["user_can_edit"] = {}
        for repo in session["user_repositories"]:
            try:
                json_file=github.get(
                    "repos/"
                    + app.config["PERSISTENCE_CONFIG"]["REPOSITORY_OWNER"] + "/"
                    + repo + "/contents/properties/properties.json",
                    hooks=dict(response=log_api_rate)
                )
                logger.debug("repo: "+repo+" - properties: "+str(json.loads(str(b64decode(json_file["content"]), "utf-8"))))
                session["properties"][repo] = json.loads(str(b64decode(json_file["content"]), "utf-8"))["property_list"]
            except GitHubError as ghe:
                logger.debug(
                    "GitHub Error trying to get the property list. We'll assume " +
                    "there is none and use default list as defined in config.py"
                )
                session["properties"][repo] = app.config["PROPERTY_LIST"]
            session["user_can_edit"][repo] = True
        logger.debug("User can edit: "+str(session["user_can_edit"]))
    except GitHubError as ghe:
        logger.error(
            "GitHubError trying to get the list of repository for user "
            + session["user_login"]
        )
        logger.error(ghe.response.text)
    return redirect(url_for('home.index'))


@module.route('/catedit-logout')
def logout():
    """
        Function that manages authentication (Github), logout

        Note: if you want to switch github users, you will have to logout of
        Github, else when logging back in, github will send the app the
        same oauth code
    """
    session["user_logged"] = None
    session["user_code"] = None
    session["user_login"] = None
    session["user_can_edit"] = None
    session["modified_categories"] = {
        repo: {} for
        repo in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]
    }
    session["deleted_categories"] = {
        repo: {} for repo
        in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]
    }
    session["properties"] = {
        repo: {} for repo
        in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]
    }
    return redirect(url_for('home.index'))
