"""
api.py:
contains the api that links the views (views.py) to the model (models.py) and
its persistence method (persistence.py). As it only trades rdf graphs
serializations strings, it isn't bound to one specific view
"""

from flask.ext.restful import Resource, Api, reqparse
from flask import request
from catedit import app, cache
from catedit.models import Category, CategoryManager


api = Api(app)

LOGGER = app.logger

CAT_PARSER = reqparse.RequestParser()
CAT_PARSER.add_argument('label', type=str)
CAT_PARSER.add_argument('description', type=str)
CAT_PARSER.add_argument('commit_message', type=str)
CAT_PARSER.add_argument('property_predicate', type=str, action="append")
CAT_PARSER.add_argument('property_object', type=str, action="append")
CAT_PARSER.add_argument('delete_message', type=str)


class CategoryAPI(Resource):
    """
        The API to create and edit categories, returns rdf graph serializations
        when successful
    """
    @classmethod
    @cache.memoize(timeout=3600)
    def get(cls, cat_id=None):
        """
            API to get the category of id cat_id, or if cat_id is None,
            get the list of category
        """
        cat_manager_instance = CategoryManager()
        if cat_id is not None:
            cat = cat_manager_instance.load_cat(cat_id)
            return cat.cat_graph.serialize(format='turtle').decode("utf-8")
        else:
            response = []
            for cat in cat_manager_instance.list_cat():
                response.append(cat.cat_graph.serialize(format='turtle')
                                             .decode("utf-8"))
            return response

    # update category cat_id
    @classmethod
    def put(cls, cat_id):
        """
            API to edit the category of id cat_id
        """
        args = CAT_PARSER.parse_args()
        cat_manager_instance = CategoryManager()

        new_property_list = []

        LOGGER.debug(args["property_predicate"])
        LOGGER.debug(args["property_object"])
        if args["property_predicate"] is not None and \
           args["property_object"] is not None:
            for property_predicate, property_object in zip(
                    args["property_predicate"],
                    args["property_object"]):
                if property_object:
                    property_object_to_append = property_object
                    # if URIRef category, we must prefix id with namespace
                    if (app.config["PROPERTY_LIST"]
                                  [property_predicate]
                                  ["object_type"]) == "uriref-category":
                        property_object_to_append = \
                            app.config["CATEGORY_NAMESPACE"] + property_object
                        LOGGER.debug(property_object_to_append)
                    new_property_list.append((property_predicate,
                                              property_object_to_append))
        LOGGER.debug(new_property_list)
        cat = cat_manager_instance.load_cat(cat_id)
        cat.edit_category(new_description=args["description"],
                          new_label=args["label"],
                          new_other_properties=new_property_list)
        cat_manager_instance.save_cat(cat, message=args["commit_message"])
        LOGGER.debug("put id: "+cat.cat_id)
        cache.clear()
        return cat.cat_graph.serialize(format='turtle').decode("utf-8"), 200
        # Maybe not send the whole cat back, see if it's worth it

    @classmethod
    def post(cls):
        """
            API to create a new category
        """
        args = CAT_PARSER.parse_args()
        property_list = []
        LOGGER.debug(args["property_predicate"])
        LOGGER.debug(args["property_object"])
        for property_predicate, property_object in zip(
                request.form.getlist('property_predicate'),
                request.form.getlist('property_object')):
            if property_object:
                if (app.config["PROPERTY_LIST"]
                              [property_predicate]
                              ["object_type"]) == "uriref-category":
                    property_list.append((property_predicate,
                                          app.config["CATEGORY_NAMESPACE"]
                                          + property_object))
                else:
                    property_list.append((property_predicate,
                                          property_object))
        LOGGER.debug(property_list)
        cat = Category(label=args["label"],
                       description=args["description"],
                       other_properties=property_list)
        cat_manager_instance = CategoryManager()
        cat_manager_instance.save_cat(cat, message=args["commit_message"])
        LOGGER.debug("post id: "+cat.cat_id)
        cache.clear()
        return cat.cat_graph.serialize(format='turtle').decode("utf-8"), 201

    @classmethod
    def delete(cls, cat_id):
        """
            API to delete the category of id cat_id
        """
        args = CAT_PARSER.parse_args()
        cat_manager_instance = CategoryManager()
        if args["delete_message"] is None:
            message = "Deleting category "+cat_id
        else:
            message = args["delete_message"]
        cat_manager_instance.delete_cat(cat_id, message=message)
        LOGGER.debug("delete id: "+cat_id)
        cache.clear()
        return 204

api.add_resource(CategoryAPI,
                 '/category/<string:cat_id>',
                 '/category',
                 endpoint='category')
