"""
views.py:
The views functions that handle the front-end of the application
"""

from catedit import app, github
from catedit.models import Category
from flask import render_template, request, redirect, url_for, session
from flask.ext.github import GitHubError
from flask_wtf import Form
from catedit.api import CategoryAPI
from wtforms import StringField, TextAreaField
from wtforms.validators import DataRequired
from rdflib import Graph
from io import StringIO

LOGGER = app.logger


class NewCategoryMinimalForm(Form):
    """
        Custom form class for creating a category with the absolute minimal
        attributes (label and description)
    """
    label = StringField(
        "Nom de la categorie (obligatoire)",
        validators=[DataRequired()]
        )
    description = TextAreaField(
        "Description de la categorie (obligatoire)",
        validators=[DataRequired()]
        )
    commit_message = StringField(
        "Message de soumission (Obligatoire)",
        validators=[DataRequired()]
        )


@app.route('/catrecap/delete-<delete_cat_id>', methods=['POST'])
@app.route('/', defaults={'delete_cat_id': None}, methods=['GET'])
@app.route('/catrecap', defaults={'delete_cat_id': None}, methods=['GET'])
def cat_recap(delete_cat_id):
    """
        View that has a list of all categories available. Template is
        catrecap.html, located in src/templates/

        Note: it also handles category deletion from the same page.
    """
    cat_api_instance = CategoryAPI()
    if delete_cat_id is None:
        LOGGER.debug("Category to delete is None")
        serialized_cat_list = cat_api_instance.get()
        # LOGGER.debug(serialized_cat_list)
        cat_list = []
        for serialized_cat in serialized_cat_list:
            cat_rdf_graph = Graph()
            cat_rdf_graph.parse(source=StringIO(serialized_cat),
                                format='turtle')
            cat = Category(graph=cat_rdf_graph)

            cat_list.append({"cat_label": cat.label,
                             "cat_description": cat.description,
                             "cat_id": cat.cat_id,
                             "cat_properties": cat.properties})
            # LOGGER.debug(c.properties)
        return render_template('catrecap.html',
                               cat_list=cat_list)
    else:
        LOGGER.debug("Category "+delete_cat_id+" will be deleted.")
        cat_api_instance.delete(delete_cat_id)
        return redirect(url_for('cat_recap'))


@app.route('/cateditor', methods=['GET', 'POST'])
@app.route('/cateditor/<string:cat_id>', methods=['GET', 'POST'])
def cat_editor(cat_id=None):
    """
        View that handles creation and edition of categories. Template is
        cateditor.html, located in src/templates
    """
    cat_api_instance = CategoryAPI()

    serialized_cat_list = cat_api_instance.get()
    cat_list = []
    for serialized_cat in serialized_cat_list:
        cat_rdf_graph = Graph()
        cat_rdf_graph.parse(source=StringIO(serialized_cat),
                            format='turtle')
        cat = Category(graph=cat_rdf_graph)

        cat_list.append({"cat_label": cat.label,
                         "cat_description": cat.description,
                         "cat_id": cat.cat_id,
                         "cat_properties": cat.properties})

    if cat_id is not None:
        specific_serialized_cat = cat_api_instance.get(cat_id)

        cat_rdf_graph = Graph()
        cat_rdf_graph.parse(source=StringIO(specific_serialized_cat),
                            format='turtle')
        cat = Category(graph=cat_rdf_graph)

        setattr(NewCategoryMinimalForm,
                'label',
                StringField("Nom de la categorie",
                            validators=[DataRequired()],
                            default=cat.label))
        setattr(NewCategoryMinimalForm,
                'description',
                TextAreaField("Description de la categorie",
                              validators=[DataRequired()],
                              default=cat.description))
        LOGGER.debug("CatForm fields preset to "
                     + cat.label + " and "
                     + cat.description)

        cat_form = NewCategoryMinimalForm(request.form)

        # GET + cat_id = Edit cat form
        if request.method == 'GET':
            return render_template('cateditor.html',
                                   cat_id=cat.cat_id,
                                   cat_properties=cat.properties,
                                   form=cat_form,
                                   cat_list=cat_list)

        # PUT + cat_id = Submit edited cat
        if cat_form.validate_on_submit():
            cat_api_instance.put(cat_id)
            return redirect(url_for('cat_recap'))
        else:
            return render_template('cateditor.html',
                                   cat_id=cat_id,
                                   cat_properties=cat.properties,
                                   form=cat_form,
                                   cat_list=cat_list)

    else:
        setattr(NewCategoryMinimalForm,
                'label',
                StringField("Nom de la categorie",
                            validators=[DataRequired()]))
        setattr(NewCategoryMinimalForm,
                'description',
                TextAreaField("Description de la categorie",
                              validators=[DataRequired()]))

        cat_form = NewCategoryMinimalForm(request.form)

        # GET seul = Create cat form
        if request.method == 'GET':
            return render_template('cateditor.html',
                                   form=cat_form,
                                   cat_list=cat_list)

        # POST seul = Submit created cat
        if cat_form.validate_on_submit():
            cat_api_instance.post()
            return redirect(url_for('cat_recap'))
        else:
            return render_template('cateditor.html',
                                   form=cat_form,
                                   cat_list=cat_list)


@app.route('/catedit-github-login')
def github_login():
    """
        Function that manages authentication (Github), login

        Note: If Persistence is set to PersistenceToFile (categories stored
        in local files, used for debugging), creates a mock user named
        "FileEditUser"
    """
    if app.config["PERSISTENCE_METHOD"] == "PersistenceToGithub":
        return github.authorize(scope="repo")
    elif app.config["PERSISTENCE_METHOD"] == "PersistenceToFile":
        session["user_logged"] = True
        session["user_can_edit"] = True
        session["user_login"] = "FileEditUser"
        return redirect(url_for('cat_recap'))


@app.route('/catedit-github-callback')
@github.authorized_handler
def github_callback(oauth_code):
    """
        Function that handles callback from Github after succesful login
    """
    session.permanent = False
    session["user_code"] = oauth_code
    session["user_logged"] = True
    session["user_login"] = github.get("user")["login"]
    try:
        repo_list = []
        repo_list = github.get("user/repos")
        for repo in repo_list:
            LOGGER.debug(repo["name"])
        session["user_can_edit"] = True
        if not any(repo["name"] == app.config["REPOSITORY_NAME"]
                   for repo in repo_list):
            session["user_can_edit"] = False
        LOGGER.debug(session["user_can_edit"])
    except GitHubError:
        LOGGER.debug("error getting repos!")

    LOGGER.debug(session["user_login"])
    return redirect(url_for('cat_recap'))


@github.access_token_getter
def token_getter():
    """
        Utility function for github-flask module to get user token when
        making authenticated requests
    """
    if session.get("user_logged", None):
        # LOGGER.debug("I made an authentified request")
        return session["user_code"]


@app.route('/catedit-logout')
def logout():
    """
        Function that manages authentication (Github), logout

        Note: if you want to switch github users, you will have to logout of
        Github, else when logging back in, github will send the app the
        same oauth code
    """
    session["user_code"] = None
    session["user_logged"] = None
    session["user_login"] = None
    session["user_can_edit"] = None
    return redirect(url_for('cat_recap'))
