"""
views.py:
The views functions that handle the front-end of the application
"""

from catedit import app, github, cache
from catedit.models import Category
from requests import get
from requests.auth import HTTPBasicAuth
import catedit.persistence
from flask import render_template, request, redirect, url_for, session, abort
from flask.ext.github import GitHubError
from flask_wtf import Form
from catedit.resources import CategoryAPI, CategoryChangesAPI
from wtforms import StringField, TextAreaField, FormField, PasswordField, \
                    FieldList, HiddenField
from wtforms.validators import DataRequired, Optional, AnyOf
from rdflib import Graph
from io import StringIO

logger = app.logger

@app.route('/', methods=['GET'])
@app.route('/index', methods=['GET'])
def cat_index():
    """
        View for index page (for now it's only a readme text so no computing
        is required)
    """
    return render_template("catindex.html")


@app.route('/<string:repository>/catrecap',
           defaults={'deleted_cat_id': None, 'deleted_modifs_id': None},
           methods=['GET', 'POST'])
@app.route('/<string:repository>/catrecap/delete-modifs-<deleted_modifs_id>',
           defaults={'deleted_cat_id': None},
           methods=['POST'])
@app.route('/<string:repository>/catrecap/delete-<deleted_cat_id>',
           defaults={'deleted_modifs_id': None},
           methods=['POST'])
def cat_recap(repository, deleted_cat_id, deleted_modifs_id):
    """
        View that has a list of all categories available. Template is
        catrecap.html, located in src/templates/

        Note: it also handles category deletion from the same page.
    """
    if repository not in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]:
        abort(404)

    cat_api_instance = CategoryAPI()
    cat_changes_api_instance = CategoryChangesAPI()

    if request.method == "POST":
        if (session.get("user_logged", None) is not None and
                session.get("user_can_edit", False) is not False):
            if deleted_modifs_id is None and deleted_cat_id is None:
                cat_changes_api_instance.delete(repository=repository)
            elif deleted_modifs_id is not None:
                logger.debug("Changes for category "
                             + deleted_modifs_id
                             + " will be deleted.")
                cat_changes_api_instance.delete(
                    repository=repository,
                    modified_cat_id=deleted_modifs_id
                )
            # We identify if we want to delete a category
            elif deleted_cat_id is not None:
                logger.debug("Category "
                             + deleted_cat_id
                             + " will be deleted.")
                cat_api_instance.delete(
                    repository=repository,
                    deleted_cat_id=deleted_cat_id
                )
        return redirect(url_for('cat_recap', repository=repository))
    elif request.method == "GET":

        deleted_cat_dict = {}
        modified_cat_dict = {}
        serialized_cat_list = []
        if session.get("user_logged", None) is not None:
            serialized_cat_list = cat_api_instance.get(repository=repository) \
                                                      [0]
            cat_changes = cat_changes_api_instance.get(repository=repository) \
                                                      [0]
            modified_cat_dict = cat_changes["modified_categories"]
            deleted_cat_dict = cat_changes["deleted_categories"]
        # logger.debug(serialized_cat_list)
        cat_list = []
        original_cat_list = []
        for serialized_cat in serialized_cat_list:
            cat_rdf_graph = Graph()
            cat_rdf_graph.parse(source=StringIO(serialized_cat),
                                format='turtle')
            original_cat_list.append(Category(graph=cat_rdf_graph))

        edited_cat_list = []
        for modified_cat_name in modified_cat_dict.keys():
            new_cat_rdf_graph = Graph()
            new_cat_rdf_graph.parse(
                source=StringIO(
                    modified_cat_dict[modified_cat_name]
                ),
                format='turtle'
            )
            edited_cat_list.append(Category(graph=new_cat_rdf_graph))
        # first we find the untouched, edited and deleted categories
        cat_state = ""
        for category in original_cat_list:
            if category.cat_id not in modified_cat_dict.keys():
                if category.cat_id in deleted_cat_dict.keys():
                    cat_state = "deleted"
                else:
                    cat_state = "untouched"

                cat_list.append({"cat_label": category.label,
                                 "cat_description": category.description,
                                 "cat_id": category.cat_id,
                                 "cat_properties": category.properties,
                                 "state": cat_state})

        # now we must find the not yet submitted categories that were created
        cat_state = ""
        logger.debug("Edited cat list: "
                     + str([cat.label for cat in edited_cat_list])
                     + " - Original cat list: "
                     + str([cat.label for cat in original_cat_list]))
        for category in edited_cat_list:
            if category.cat_id not in [cat.cat_id for
                                       cat in original_cat_list]:
                cat_state = "created"
            else:
                cat_state = "modified"
            cat_list.append({"cat_label": category.label,
                             "cat_description": category.description,
                             "cat_id": category.cat_id,
                             "cat_properties": category.properties,
                             "state": cat_state})

            # logger.debug(c.properties)
        cat_list = sorted(cat_list, key=lambda cat: cat['cat_id'])
        return render_template('catrecap.html',
                               cat_list=cat_list,
                               current_repository=repository)

class CommitForm(Form):
    """
        Custom form class for commiting changes
    """
    commit_message = StringField(
        "Message de soumission (obligatoire)",
        validators=[DataRequired()]
    )


@app.route('/<string:repository>/catmodifs/delete-modifs-<deleted_modifs_id>',
           defaults={'deleted_cat_id': None},
           methods=['POST'])
@app.route('/<string:repository>/catmodifs/delete-<deleted_cat_id>',
           defaults={'deleted_modifs_id': None},
           methods=['POST'])
@app.route('/<string:repository>/catmodifs',
           defaults={'deleted_cat_id': None, 'deleted_modifs_id': None},
           methods=['GET', 'POST'])
def cat_modifs(repository, deleted_cat_id, deleted_modifs_id):
    """
        View that handles all the visualisation of changes for a user's
        session, links to the editor forms, allows the users to cancel their
        own changes and submits all their changes.
    """
    if repository not in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]:
        abort(404)

    logger.debug(repository)

    changes_list = []

    existing_cat_list = []
    modified_cat_list = []
    created_cat_list = []
    total_cat_list = []

    deleted_cat_namelist = []
    modified_cat_namelist = []

    cat_api_instance = CategoryAPI()
    cat_changes_api_instance = CategoryChangesAPI()

    if deleted_cat_id is None and deleted_modifs_id is None:

        commit_form = CommitForm(request.form)

        # if it's a GET with no delete_cat_id or deleted_modifs_id, then we'll
        # display the changes
        if request.method == "GET":
            if session.get("user_logged", None) is not None:
                serialized_cat_list = cat_api_instance.get(
                    repository=repository
                )[0]
                changes_list = cat_changes_api_instance.get(
                    repository=repository
                )[0]

                cat_data = dict({})

                # Creating existing cat list
                for serialized_cat in serialized_cat_list:
                    cat_rdf_graph = Graph()
                    cat_rdf_graph.parse(source=StringIO(serialized_cat),
                                        format='turtle')
                    cat = Category(graph=cat_rdf_graph)

                    cat_data = {
                        "cat_label": cat.label,
                        "cat_description": cat.description,
                        "cat_id": cat.cat_id,
                        "cat_properties": cat.properties
                    }

                    existing_cat_list.append(cat_data)
                    total_cat_list.append(cat_data)
                    cat_data = dict({})

                # Creating modified and created cat lists
                for modified_cat_name in \
                        changes_list.get("modified_categories", {}).keys():
                    modified_cat_rdf_graph = Graph()
                    modified_cat_rdf_graph.parse(
                        source=StringIO(
                            changes_list["modified_categories"]
                                        [modified_cat_name]
                        ),
                        format='turtle'
                    )
                    modified_cat = Category(graph=modified_cat_rdf_graph)

                    cat_data = {
                        "cat_label": modified_cat.label,
                        "cat_description": modified_cat.description,
                        "cat_id": modified_cat.cat_id,
                        "cat_properties": modified_cat.properties
                    }
                    if modified_cat.cat_id in \
                            [existing_cat["cat_id"]
                                for existing_cat in existing_cat_list]:
                        modified_cat_list.append(cat_data)
                    else:
                        created_cat_list.append(cat_data)
                        total_cat_list.append(cat_data)
                    cat_data = dict({})

                # Creating deleted cat list
                deleted_cat_namelist = list(
                    changes_list["deleted_categories"].keys()
                )
                modified_cat_namelist = list(
                    changes_list["modified_categories"].keys()
                )
                logger.debug("total:"+str(total_cat_list))
                logger.debug("existing:"+str(existing_cat_list))
                logger.debug("created:"+str(created_cat_list))
                logger.debug("modified:"+str(modified_cat_list))
                logger.debug("modified names:"+str(modified_cat_namelist))
                logger.debug("deleted names:"+str(deleted_cat_namelist))

            return render_template('catmodifs.html',
                                   total_cat_list=total_cat_list,
                                   existing_cat_list=existing_cat_list,
                                   created_cat_list=created_cat_list,
                                   modified_cat_list=modified_cat_list,
                                   modified_cat_namelist=modified_cat_namelist,
                                   deleted_cat_namelist=deleted_cat_namelist,
                                   commit_form=commit_form,
                                   current_repository=repository)
        # If it's a POST with no delete_cat_id and delete_modifs_id, then we
        # will submit all the stored changes
        elif request.method == "POST":
            if commit_form.validate_on_submit():
                cat_api_instance.put(repository=repository)
                cat_changes_api_instance.delete(repository=repository)
            return redirect(url_for('cat_recap', repository=repository))

    # One of the ids is not None (either deleting a category or a modification)
    else:
        # We only do that if we have a POST
        if request.method == "POST":
            # We identify if we want to delete a change
            if (session.get("user_logged", None) is not None) \
                and \
               (session.get("user_can_edit", False) is not False):
                if deleted_modifs_id is not None:
                    cat_changes_api_instance.delete(
                        repository=repository,
                        modified_cat_id=deleted_modifs_id
                    )
                # We identify if we want to delete a category
                elif deleted_cat_id is not None:
                    cat_api_instance.delete(
                        repository=repository,
                        deleted_cat_id=deleted_cat_id
                    )
        return redirect(url_for('cat_modifs', repository=repository))



class PropertyForm(Form):
    """
        Form of a given property, each one is a couple of hidden fields that
        can be Javascript-generated in the template
    """
    property_predicate = HiddenField(
        validators=[AnyOf(values=app.config["PROPERTY_LIST"].keys())]
    )
    property_object = HiddenField(
        validators=[DataRequired()]
    )


class CategoryForm(Form):
    """
        Custom form class for creating a category with the absolute minimal
        attributes (label and description)
    """
    label = StringField(
        "Nom de la categorie (obligatoire)",
        validators=[DataRequired()]
    )
    description = TextAreaField(
        "Description de la categorie (obligatoire)",
        validators=[DataRequired()]
    )
    properties = FieldList(FormField(PropertyForm), validators=[Optional()])


@app.route('/<string:repository>/cateditor',
           methods=['GET', 'POST'])
@app.route('/<string:repository>/cateditor/<string:cat_id>',
           methods=['GET', 'POST'])
def cat_editor(repository, cat_id=None):
    """
        View that handles creation and edition of categories. Template is
        cateditor.html, located in src/templates
    """
    if repository not in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]:
        abort(404)

    cat_api_instance = CategoryAPI()
    cat_changes_api_instance = CategoryChangesAPI()

    specific_serialized_cat = ""
    # Serialization of the category of id cat_id, either from
    # CategoryChangesAPI (if it was modified) or from CategoryAPI (if it
    # was not)

    changes_response = None
    # Changes on the category cat_id we get from CategoryChangesAPI

    current_cat = None
    current_cat_id = None
    current_cat_properties = []
    # Args for the template, if we create a new category they are not used

    cat_list = []
    # Category list that will be used by the template
    deleted_cat_dict = {}
    # Deleted categories we won't append to cat_list
    modified_cat_dict = {}
    # Modified categories to append to cat_list in case label changed
    serialized_cat_list = []
    # Existing categories we get from CategoryAPI
    cat_changes = {}
    # Changes list we get from CategoryChangesAPI

    if cat_id is not None:
        if session.get("user_logged", None) is not None:
            changes_response = cat_changes_api_instance.get(
                repository=repository,
                modified_cat_id=cat_id
            )
            if changes_response[0]["type"] is "deleted":
                abort(404)
            elif changes_response[0]["type"] is "modified":
                specific_serialized_cat = changes_response[0] \
                                                          ["category"] \
                                                          [cat_id]
            elif changes_response[0]["type"] is "untouched":
                specific_serialized_cat = cat_api_instance.get(
                    repository=repository,
                    cat_id=cat_id
                )[0]
        logger.debug(specific_serialized_cat)

        cat_rdf_graph = Graph()
        cat_rdf_graph.parse(source=StringIO(specific_serialized_cat),
                            format='turtle')

        current_cat = Category(graph=cat_rdf_graph)
        current_cat_id = current_cat.cat_id
        current_cat_properties = current_cat.properties
        logger.debug(current_cat.properties)

    serialized_cat_list = cat_api_instance.get(repository)[0]
    cat_changes = cat_changes_api_instance.get(repository)[0]
    deleted_cat_dict = cat_changes["deleted_categories"]
    modified_cat_dict = cat_changes["modified_categories"]
    logger.debug(changes_response)

    for modified_cat_name in modified_cat_dict.keys():
        modified_cat_rdf_graph = Graph()
        modified_cat_rdf_graph.parse(
            source=StringIO(modified_cat_dict[modified_cat_name]),
            format='turtle'
        )
        modified_cat = Category(graph=modified_cat_rdf_graph)
        cat_list.append({"cat_label": modified_cat.label,
                         "cat_description": modified_cat.description,
                         "cat_id": modified_cat.cat_id,
                         "cat_properties": modified_cat.properties})

    for serialized_cat in serialized_cat_list:
        cat_rdf_graph = Graph()
        cat_rdf_graph.parse(source=StringIO(serialized_cat),
                            format='turtle')
        cat = Category(graph=cat_rdf_graph)
        if cat.cat_id not in deleted_cat_dict.keys() and \
           cat.cat_id not in modified_cat_dict.keys():
            cat_list.append({"cat_label": cat.label,
                             "cat_description": cat.description,
                             "cat_id": cat.cat_id,
                             "cat_properties": cat.properties})

    cat_form = CategoryForm(request.form)

    if request.method == "GET":
        """
            GET Method means we will display the editor
        """
        if current_cat is not None:
            cat_form.label.data = current_cat.label
            cat_form.description.data = current_cat.description
            for (cat_predicate, cat_object) in current_cat_properties:
                if app.config["PROPERTY_LIST"] \
                             [cat_predicate] \
                             ["object_type"] == "uriref-category":
                    namespace, object_id = cat_object.split("#",1)
                else:
                    object_id = cat_object
                cat_form.properties.append_entry({
                    "property_predicate": cat_predicate,
                    "property_object": object_id
                })
        # list of category that will be used in property editor, list of dict
        # {"cat_id": cat.cat_id,
        #  "cat_label": cat.label,
        #  "cat_description": cat.description,
        #  "cat_properties": cat.properties}
        logger.debug("category list that can be linked :" + str(cat_list))
        logger.debug(
            "deleted categories list that can't be linked :"
            + str(list(deleted_cat_dict.keys()))
        )
        return render_template('cateditor.html',
                               cat_id=current_cat_id,
                               cat_properties=current_cat_properties,
                               form=cat_form,
                               cat_list=cat_list,
                               deleted_cat_list=list(deleted_cat_dict.keys()),
                               current_repository=repository)
    elif request.method == "POST":
        """
            POST Method means we will compute the form data, call the relevant
            "POST" and "PUT" method on CategoryAPI, and return to a display
            page (catrecap for now, ideally the one from whence we came)
        """
        if (session.get("user_logged", None) is not None and
                session.get("user_can_edit", False) is not False):

            if cat_form.validate_on_submit():
                cat_data = {}
                cat_data["label"] = cat_form.label.data
                cat_data["description"] = cat_form.description.data
                cat_data["properties"] = [
                    (cat_property["property_predicate"],
                     cat_property["property_object"])
                    for cat_property in cat_form.properties.data
                ]

                if cat_id is not None:
                    logger.debug(str(cat_data))
                    cat_api_instance.put(
                        repository=repository,
                        cat_id=cat_id,
                        cat_data=cat_data
                    )
                else:
                    logger.debug(str(cat_data))
                    cat_api_instance.post(
                        repository=repository,
                        cat_data=cat_data
                    )
                return redirect(url_for('cat_recap', repository=repository))
            else:
                # if form doesn't validate we don't want to delete whatever
                # changes the user did
                return render_template('cateditor.html',
                                       cat_id=cat_id,
                                       cat_properties=current_cat_properties,
                                       form=cat_form,
                                       cat_list=cat_list,
                                       deleted_cat_list=list(
                                           deleted_cat_dict.keys()
                                       ),
                                       current_repository=repository)

        # If user wasn't logged or couldn't edit but somehow submitted a POST
        else:
            return redirect(url_for('cat_index'))


class LoginForm(Form):
    """
        Custom form class for commiting changes
    """
    user_login = StringField(
        "Nom d'utilisateur Github",
        validators=[DataRequired()]
    )
    user_password = PasswordField(
        "Mot de passe Github",
        validators=[DataRequired()]
    )


@app.route('/catedit-login', methods=["GET", "POST"])
def github_login():
    """
        Function that manages authentication (Github), login

        Note: If Persistence is set to PersistenceToFile (categories stored
        in local files, used for debugging), creates a mock user named
        "FileEditUser"
    """
    if not session.get("user_logged", False):
        session["modified_categories"] = {
            repo: {} for repo in app.config["PERSISTENCE_CONFIG"]
                                           ["REPOSITORY_LIST"]
        }
        session["deleted_categories"] = {
            repo: {} for repo in app.config["PERSISTENCE_CONFIG"]
                                           ["REPOSITORY_LIST"]
        }
        if app.config["PERSISTENCE_CONFIG"]["METHOD"] == "PersistenceToGithub":
            login_form = LoginForm(request.form)
            if request.method == "POST":
                if login_form.validate_on_submit():
                    # We'll try to get the auth token for given username
                    attempted_login = login_form.user_login.data
                    try:
                        auth_response = get(
                            "https://api.github.com/"
                            + "authorizations",
                            auth=HTTPBasicAuth(
                                login_form.user_login.data,
                                login_form.user_password.data
                            )
                        )
                        for auth in auth_response.json():
                            if auth["app"]["client_id"] \
                            == app.config["GITHUB_CLIENT_ID"]:
                                session["user_code"] = auth["token"]
                                session["user_logged"] = True
                    except:
                        logger.debug(
                            "Error requesting authorizations for"
                            + " user. Either the user is new to catedit, or "
                            + "entered a wrong username/password"
                        )
                    logger.debug(
                        "user token found by request: "
                        + str(session.get("user_code", None))
                    )
                    if session.get("user_code", None) == None:
                        # We didn't get it, so we direct the user to the login page
                        # with a link to github oauth system
                        return render_template(
                            'catlogin.html',
                            form = login_form
                        )
                    else:
                        # we did get it, so we redirect to callback function
                        # to wrap up user auth
                        return redirect(url_for('github_callback'))
                else:
                    # form didn't validate, so we send it back to user
                    return render_template(
                        'catlogin.html',
                        form = login_form
                    )
            elif request.method == "GET":
                # We'll render the login form
                return render_template(
                    'catlogin.html',
                    form=login_form,
                )
        elif app.config["PERSISTENCE_CONFIG"]["METHOD"] == "PersistenceToFile":
            session["user_logged"] = True
            session["user_can_edit"] = {}
            session["user_can_edit"]["local"] = True
            session["user_login"] = "FileEditUser"
            return redirect(url_for('cat_index'))
    else:
        return redirect(url_for('cat_index'))

@app.route('/catedit-login-confirm', methods=["GET", "POST"])
def github_login_confirm():
    """
        Function called if the user is new or revoked the auth token
    """
    if not session.get("user_logged", False):
        if request.method == "POST":
            return github.authorize(
                scope="repo",
                redirect_uri=url_for('github_callback', _external=True)
            )
    else:
        return redirect(url_for('cat_index'))

@app.route('/catedit-github-callback')
@github.authorized_handler
def github_callback(oauth_code):
    """
        Function that handles callback from Github after succesful login
    """
    session.permanent = False
    if session.get("user_code", None) == None:
    # That means we got here using github callback and not the login form
        session["user_code"] = oauth_code
    logger.debug(session["user_code"])
    session["user_logged"] = True
    session["user_login"] = "auth-error"
    try:
        logger.debug(
            "after login: "
            + str(github.get("rate_limit")["resources"])
        )
        session["user_login"] = github.get("user")["login"]
    except GitHubError as ghe:
        logger.error(
            "GitHubError trying to get the user login"
        )
        logger.error(ghe.response.text)
    try:
        repo_list = []
        repo_list = github.get("user/repos")
        logger.debug(str(github.get("rate_limit")["resources"]))
        for repo in repo_list:
            logger.debug(repo["name"])
        user_repos_name = [repo["name"] for repo in repo_list]
        logger.debug(
            str(user_repos_name) + " "
            + str(app.config["PERSISTENCE_CONFIG"]
                            ["REPOSITORY_LIST"])
        )
        session["user_repositories"] = list(
            set(user_repos_name).intersection(
                app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]
            )
        )
        session["user_can_edit"] = {}
        for repo in session["user_repositories"]:
            if repo in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]:
                session["user_can_edit"][repo] = True
        logger.debug(session["user_can_edit"])
    except GitHubError as ghe:
        logger.error(
            "GitHubError trying to get the list of repository for user "
            + session["user_login"]
        )
        logger.error(ghe.response.text)
    return redirect(url_for('cat_index'))


@github.access_token_getter
def token_getter():
    """
        Utility function for github-flask module to get user token when
        making authenticated requests
    """
    if session.get("user_logged", None):
        # logger.debug("I made an authentified request")
        return session["user_code"]


@app.route('/catedit-logout')
def logout():
    """
        Function that manages authentication (Github), logout

        Note: if you want to switch github users, you will have to logout of
        Github, else when logging back in, github will send the app the
        same oauth code
    """
    session["user_logged"] = None
    session["user_code"] = None
    session["user_login"] = None
    session["user_can_edit"] = None
    session["modified_categories"] = {
        repo: {} for repo in app.config["PERSISTENCE_CONFIG"]
                                       ["REPOSITORY_LIST"]
    }
    session["deleted_categories"] = {
        repo: {} for repo in app.config["PERSISTENCE_CONFIG"]
                                       ["REPOSITORY_LIST"]
    }
    return redirect(url_for('cat_index'))
