"""
persistence.py:
contains what we need to make a (category) objects (see models.py) persist
beyond application scope and load them from outside sources

For now we can save, load and list
* to/from a file (using a turtle rdf graph serialization for categories)
* to/from a github repository (using a turtle rdf graph serialization
for categories)
"""
from abc import ABCMeta, abstractmethod
from catedit import app, github
from base64 import b64decode
from flask import session
from flask.ext.github import GitHubError
import os
import json

logger = app.logger


class Persistence:
    """
        Meta-class for all persistence classes
    """
    __metaclass__ = ABCMeta

    @abstractmethod
    def session_compliant(self):
        """
            Abstract - Can this persistence submit a changeset?
        """
        return

    @abstractmethod
    def submit_changes(self, **kwargs):
        """
            Abstract - Submit all saved objects, only useful if persistence
            method can submit a changeset
        """
        return

    @abstractmethod
    def save(self, **kwargs):
        """
            Abstract - Saves object
        """
        return

    @abstractmethod
    def delete(self, **kwargs):
        """
            Abstract - Deletes object
        """
        return

    @abstractmethod
    def load(self, **kwargs):
        """
            Abstract - Loads object
        """
        return

    @abstractmethod
    def list(self, **kwargs):
        """
            Abstract - Lists objects
        """
        return


class PersistenceToFile(Persistence):
    """
        Persistence Class for saving to a local file (see config.py
        for tweaking)

        Expected kwargs for saving to/loading from a file are:
        * name : name of the file to write in/read from
        * content : desired content of the file when writing
    """
    @property
    def session_compliant(self):
        """
            Not session compliant: each modification is submitted
        """
        return False

    def submit_changes(self, **kwargs):
        """
            As each modification is submitted, this is where we save to a file
        """

    def save(self, **kwargs):
        """
            Saves to a file
        """
        path_to_save = app.config["FILE_SAVE_DIRECTORY"]+kwargs["name"]
        file_to_save = open(path_to_save, 'wb')
        file_to_save.write(kwargs["content"])
        file_to_save.close()

    def load(self, **kwargs):
        """
            Loads from a file
        """
        path_to_load = app.config["FILE_SAVE_DIRECTORY"]+kwargs["name"]
        file_to_load = open(path_to_load, 'rb')
        file_content = file_to_load.read()
        file_to_load.close()
        return file_content

    def delete(self, **kwargs):
        """
            Deletes a file
        """
        path_to_delete = app.config["FILE_SAVE_DIRECTORY"]+kwargs["name"]
        os.remove(path_to_delete)

    # IDEA: return { file_name: file_content } type dict
    def list(self, **kwargs):
        """
            Lists all files in file directory (as set in config.py)
        """
        file_content_list = []
        for file_name in os.listdir(app.config["FILE_SAVE_DIRECTORY"]):
            if not file_name or file_name[0] == ".":
                continue
            path_to_load = open(app.config["FILE_SAVE_DIRECTORY"]+file_name)
            file_content = path_to_load.read()
            path_to_load.close()
            file_content_list.append(file_content)
        # logger.debug(file_content_list)
        return file_content_list


class PersistenceToGithub(Persistence):
    """
        Persistence Class for saving to/loading from a Github repository (see
        config.py for tweaks)

        Expected kwargs for saving to/loading from a Github repository are:
        * name : name of the file to write in/read from
        * content : desired content of the file when writing
        * message : when saving or deleting, commit message to be saved
        to Github
    """
    @property
    def session_compliant(self):
        """
            Not session compliant: each modification is comitted
        """
        return True

    def submit_changes(self, **kwargs):
        """
            Saves all the recorded files in the session dict to a Github
            repository

            Expected kwargs is:
            * message: the commit message to document the changes

            IMPORTANT: To save to a file Github, we have to use Git
            internal mechanics:
            1) Get the current reference for master branch
            2) Get the latest commit on that reference
            3) Get the tree associated to this commit
            4) Create a new tree using the old tree and the session dict that
            contains all the changes (keys are "modified_categories" and
            "deleted_categories")
                4-1) This requires creating new blobs for new files
            5) Create a new commit referencing the previous commit as parent
            and the new tree we just created
            6) Update the master branch reference so it points on this new
            commit

            About point 4):
            We have 3 list of elements: files already in repo, modified files,
            and deleted_files
            First, we loop on the files already in repo: this allows us to
            update it (if it's in both original file list and modified files
            list) or delete it (if it's in both original file list and deleted
            files list)
            Then, we loop on the modified files list. If it isn't in the
            original file list, then it's a new file and we append it to the
            tree
        """

        # point 1
        ref_master = github.get("repos/"
                                + app.config["REPOSITORY_OWNER"] + "/"
                                + app.config["REPOSITORY_NAME"]
                                + "/git/refs/heads/master")
        logger.debug(str(ref_master))
        # point 2
        last_commit_master = github.get("repos/"
                                        + app.config["REPOSITORY_OWNER"] + "/"
                                        + app.config["REPOSITORY_NAME"]
                                        + "/git/commits/"
                                        + ref_master["object"]["sha"])

        logger.debug(str(last_commit_master))
        # point 3
        last_commit_tree = github.get("repos/"
                                      + app.config["REPOSITORY_OWNER"] + "/"
                                      + app.config["REPOSITORY_NAME"]
                                      + "/git/trees/"
                                      + last_commit_master["tree"]["sha"]
                                      + "?recursive=1")

        logger.debug(str(last_commit_tree))
        # point 4
        new_tree_data = {"tree": []}
        for element in last_commit_tree["tree"]:
            logger.debug(element)
            if element["type"] == "blob":
                # test if element is in deleted categories, if it is,
                # no point doing anything, the file won't be in the new tree
                if not(
                    element["path"] in [
                        (app.config["CATEGORIES_PATH"] + category["name"])
                        for category in session.get("deleted_categories", [])
                    ]
                ):

                    # the element is either modified or untouched so in both
                    # case we'll need a blob sha
                    blob_sha = ""

                    # test if element is in modified categories
                    if (
                        element["path"] in [
                            (app.config["CATEGORIES_PATH"] + category["name"])
                            for category in
                            session.get("modified_categories", [])
                        ]
                    ):
                        # find element in modified categories
                        for category in session["modified_categories"]:
                            if element["path"] == (
                                app.config["CATEGORIES_PATH"] + category["name"]
                            ):
                                # 4-1 for edited files
                                new_blob_data = {
                                    "content": category["content"],
                                    "encoding": "utf-8"
                                }
                                new_blob = github.post(
                                    "repos/"
                                    + app.config["REPOSITORY_OWNER"] + "/"
                                    + app.config["REPOSITORY_NAME"]
                                    + "/git/blobs",
                                    data=new_blob_data
                                )
                                blob_sha = new_blob["sha"]
                                break

                    # this means element is an untouched file
                    else:
                        blob_sha = element["sha"]
                    new_tree_data["tree"].append({"path": element["path"],
                                                  "mode": element["mode"],
                                                  "type": "blob",
                                                  "sha": blob_sha})
        logger.debug(str(new_tree_data["tree"]))
        # Now we'll add new files in the tree
        for category in session.get("modified_categories", []):
            logger.debug(app.config["CATEGORIES_PATH"]+category["name"]
                         + " should not be in "
                         + str([elt["path"] for
                                elt in last_commit_tree["tree"]]))
            if (app.config["CATEGORIES_PATH"]+category["name"] not in
                    [file["path"] for file in last_commit_tree["tree"]]):

                # 4-1 for added files
                new_blob_data = {"content": category["content"],
                                 "encoding": "utf-8"}
                new_blob = github.post("repos/"
                                       + app.config["REPOSITORY_OWNER"] + "/"
                                       + app.config["REPOSITORY_NAME"]
                                       + "/git/blobs",
                                       data=new_blob_data)
                new_tree_data["tree"].append({
                    "path": app.config["CATEGORIES_PATH"] + category["name"],
                    "mode": "100644",
                    "type": "blob",
                    "sha": new_blob["sha"]
                })
        logger.debug(str(new_tree_data))

        # Finally, we post the new tree
        new_tree_response = github.post("repos/"
                                        + app.config["REPOSITORY_OWNER"]+"/"
                                        + app.config["REPOSITORY_NAME"]
                                        + "/git/trees",
                                        data=new_tree_data)

        # point 5
        new_commit_data = {"message": kwargs["message"],
                           "parents": [last_commit_master["sha"]],
                           "tree": new_tree_response["sha"]}
        logger.debug(str(new_commit_data))
        new_commit = github.post("repos/"
                                 + app.config["REPOSITORY_OWNER"]+"/"
                                 + app.config["REPOSITORY_NAME"]
                                 + "/git/commits",
                                 data=new_commit_data)
        logger.debug(str(new_commit))

        # point 6
        new_head_data = {"sha": new_commit["sha"], "force": "true"}
        logger.debug(str(new_head_data))
        new_head = github.patch("repos/"
                                + app.config["REPOSITORY_OWNER"] + "/"
                                + app.config["REPOSITORY_NAME"]
                                + "/git/refs/heads/master",
                                data=json.dumps(new_head_data))
        logger.debug(str(new_head))
        session["deleted_categories"] = []
        session["modified_categories"] = []

    def save(self, **kwargs):
        """
            Saves given file to the session dict

            Expected kwargs should be:
            * name: the name of the file to save
            * content: the content of the file to save
        """
        session["modified_categories"][:] = [
            elt for elt in session.get("modified_categories", [])
            if elt["name"] != kwargs["name"]
        ]
        session["modified_categories"].append(
            {"name": kwargs["name"],
             "content": str(kwargs["content"], "utf-8")}
        )
        # Now we must clean the deleted categories list in case the modified
        # category was deleted before being edited

        for element in session.get("deleted_categories", []):
            if element["name"] == kwargs["name"]:
                session["deleted_categories"].remove(element)

    def load(self, **kwargs):
        """
            Loads from a Github repository
        """
        try:
            filedict = github.get("repos/"
                                  + app.config["REPOSITORY_OWNER"]+"/"
                                  + app.config["REPOSITORY_NAME"]
                                  + "/contents/"
                                  + app.config["CATEGORIES_PATH"]
                                  + kwargs["name"])
            file_content = str(b64decode(filedict["content"]), "utf-8")
        except GitHubError as ghe:
            logger.debug("Github Error trying to get file: "+kwargs["name"])
            logger.debug("Github sent an error, if 404, either you may not \
                         have access to the repository or it doesn't exist ")
            logger.debug(ghe.response.text)
        return file_content

    def delete(self, **kwargs):
        """
            Deletes from a Github repository
            Calling delete for a file already deleted will restore it
            (by deleting it from the delete files list)
        """
        if (kwargs["name"] in [
            element["name"] for element in session.get("deleted_categories", [])
        ]):
            session["deleted_categories"].remove({"name": kwargs["name"]})
            # warning, not safe if 2 files share the same name (or category id)
            # but that shouldn't happen
        else:
            session["deleted_categories"].append({"name": kwargs["name"]})
            # now we must clean the modified categories list in case the
            # deleted category was modified before
            for element in session.get("modified_categories", []):
                if element["name"] == kwargs["name"]:
                    session["modified_categories"].remove(element)

    def list(self, **kwargs):
        """
            Lists all files in the github repository (as set in config.py)
        """
        filenames_list = []
        try:
            files_in_repo = github.get("repos/"
                                       + app.config["REPOSITORY_OWNER"]+"/"
                                       + app.config["REPOSITORY_NAME"]
                                       + "/contents/"
                                       + app.config["CATEGORIES_PATH"])
            filenames_list = [github_file["name"]
                              for github_file in files_in_repo]
            # logger.debug(filenames_list)
        except GitHubError as ghe:
            logger.debug("Github Error trying to get the file list in the \
                         category repository")
            logger.debug("NOTE: Github sent an error, if 404 either you \
                         may not have access to the repository or it doesn't \
                         exist or there isn't any files in it")
            logger.debug(ghe.response.text)

        file_content_list = []
        for filename in filenames_list:
            try:
                filedict = github.get("repos/"
                                      + app.config["REPOSITORY_OWNER"]+"/"
                                      + app.config["REPOSITORY_NAME"]
                                      + "/contents/"
                                      + app.config["CATEGORIES_PATH"]
                                      + filename)
                file_content_list.append(str(b64decode(filedict["content"]),
                                         "utf-8"))
            except GitHubError as ghe:
                logger.debug("Github Error trying to get file: "+filename)
                logger.debug(ghe.response.text)
        # logger.debug(file_content_list)
        return file_content_list
