"""
social.py:
The views functions that handle comment display and posting
"""

from catedit import app
from views.utils import check_user_status, get_comments, \
                        post_comment, get_commits, get_issues, \
                        get_category_list
from flask import render_template, request, redirect, url_for, \
                  abort, Blueprint
from flask_wtf import Form
from wtforms import TextAreaField, StringField
from wtforms.validators import DataRequired

module = Blueprint('social', __name__)
logger = app.logger


@module.route("/<string:repository>/social",
              methods=["GET"])
def index(repository):
    """
        View that displays every changeset and thread in general discussion
        and links to every thread
    """
    if repository not in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]:
        abort(404)

    check_user_status(repository)

    changeset_list = get_commits(repository)
    discussion_list = get_issues(repository)

    return render_template(
        "social/index.html",
        current_repository=repository,
        discussion_list=discussion_list,
        changeset_list=changeset_list
    )


class CommentForm(Form):
    """
        Custom form class for commiting changes
    """
    comment_field = TextAreaField(
        "Poster un commentaire: ",
        validators=[DataRequired()]
    )


@module.route("/<string:repository>/changesets/<string:changeset_id>",
              methods=["GET", "POST"])
def changeset(repository, changeset_id):
    """
        View that displays a snapshot of the repository as it was for a given
        changeset, and the related discussion to this changeset. Allows
        users to comment on it.

        For re-building the old category set we use the tree and blob data:
        1) first we get the commit of id changeset_id
        2) we get the tree associated with this commit
        3) we get all the blobs referenced by this tree
        4) we build categories from the blobs contents
        --> Will probably be incorporated into a function later on

    """
    if repository not in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]:
        abort(404)

    check_user_status(repository)

    comment_form = CommentForm()
    comments_list = get_comments({
        "repository": repository,
        "type": "commits",
        "id": changeset_id
    })
    cat_list = get_category_list(repository, changeset_id)

    if request.method == "GET":
        return render_template(
            "social/changeset.html",
            cat_list=cat_list,
            comments=comments_list,
            changeset_id=changeset_id,
            comment_form=comment_form,
            current_repository=repository
        )
    elif request.method == "POST":
        if comment_form.validate_on_submit():
            comment_data = {
                "repository": repository,
                "type": "commits",
                "thread_id": changeset_id,
                "comment_body": comment_form.comment_field.data
            }
            return_id = post_comment(comment_data)
            return redirect(url_for(
                "social.changeset",
                repository=repository,
                changeset_id=return_id
            ))
        else:
            return render_template(
                "social/changeset.html",
                cat_list=cat_list,
                comments=comments_list,
                changeset_id=changeset_id,
                comment_form=comment_form,
                current_repository=repository
            )


class NewDiscussionForm(Form):
    """
        Custom form class for commiting changes
    """
    discussion_title = StringField(
        "Titre de la discussion: ",
        validators=[DataRequired()]
    )
    comment_field = TextAreaField(
        "Commentaire d'ouverture (facultatif): "
    )


@module.route("/<string:repository>/discussions/<string:discussion_id>",
              methods=["GET", "POST"])
def discussion(repository, discussion_id):
    """
        View that displays the discussion of a given id and allows users
        to post comments on it.
    """
    if repository not in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]:
        abort(404)

    check_user_status(repository)

    comment_form = None
    if discussion_id == "new":
        comment_form = NewDiscussionForm()
        comments_list = []
    else:
        comment_form = CommentForm()
        comments_list = get_comments({
            "repository": repository,
            "type": "issues",
            "id": discussion_id
        })

    if request.method == "GET":
        return render_template(
            "social/discussion.html",
            comments=comments_list,
            comment_form=comment_form,
            current_repository=repository,
            discussion_id=discussion_id
        )
    elif request.method == "POST":
        if comment_form.validate_on_submit():
            comment_data = {
                "repository": repository,
                "type": "issues",
                "thread_id": discussion_id,
                "comment_body": comment_form.comment_field.data
            }
            if discussion_id == "new":
                comment_data["thread_title"] = comment_form.discussion_title.data
            return_id = post_comment(comment_data)
            return redirect(url_for(
                "social.discussion",
                repository=repository,
                discussion_id=return_id
            ))
        else:
            return render_template(
                "social/discussion.html",
                comments=comments_list,
                comment_form=comment_form,
                current_repository=repository,
                discussion_id=discussion_id
            )
