"""
resources.py:
contains the api that links the views (views.py) to the model (models.py) and
its persistence method (persistence.py). As it only trades rdf graphs
serializations strings, it isn't bound to one specific view
"""


from rdflib import Graph
from flask.ext.restful import Resource, reqparse
from flask import session
from catedit import app, cache
from catedit.models import Category, CategoryManager
import catedit.persistence
from io import StringIO

logger = app.logger

cat_parser = reqparse.RequestParser()
cat_parser.add_argument('label', type=str)
cat_parser.add_argument('description', type=str)
cat_parser.add_argument('property_predicate', type=str, action="append")
cat_parser.add_argument('property_object', type=str, action="append")

cat_parser.add_argument('commit_message', type=str)


class CategoryAPI(Resource):
    """
        The API to create and edit categories, returns rdf graph serializations
        when successful
    """
    @cache.memoize(timeout=3600)
    def get(self, repository, cat_id=None):
        """
            API to get the category of id cat_id, or if cat_id is None,
            get the list of category
        """
        cat_manager_instance = CategoryManager(
            getattr(
                catedit.persistence,
                app.config["PERSISTENCE_CONFIG"]["METHOD"]
            )(repository=repository),
        )
        if cat_id is not None:
            cat = cat_manager_instance.load_category(cat_id)
            if cat is not None:
                return cat.cat_graph.serialize(
                    format='turtle'
                ).decode("utf-8"), 200
            else:
                return 404
        else:
            response = []
            for cat in cat_manager_instance.list_categories():
                response.append(
                    cat.cat_graph.serialize(format='turtle').decode("utf-8")
                )
            return response, 200

    # update category cat_id
    def put(self, repository, cat_id=None, cat_data=None):
        """
            API to edit an existing category

            Args are:
            * repository: the repository where the category cat_id is stored
            * cat_id : the id of the category to edit
            * cat_data : the new data of the category, dict of the form:
            {
                "label": edited_cat_label,
                "description": edited_cat_description,
                "properties": [(predicate1, object1), (predicate2, object2)]
            }
            List of predicate is available in config.py, key PROPERTY_LIST

            Note: If cat_id is None and persistence support change sets, will
            submit all changes to category list
        """
        cat_manager_instance = CategoryManager(
            getattr(
                catedit.persistence,
                app.config["PERSISTENCE_CONFIG"]["METHOD"]
            )(repository=repository),
        )
        args = cat_parser.parse_args()
        if cat_id is None:
            if cat_manager_instance.persistence.session_compliant is True:
                logger.debug(
                    "Submitting - deleted categories are:"
                    + str(
                        session.get(
                            "deleted_categories", {}
                        ).get(repository, {})
                    ) + " and modified categories are:"
                    + str(
                        session.get(
                            "modified_categories", {}
                        ).get(repository, {})
                    )
                )
                cat_manager_instance.save_changes(
                    deleted_cat_dict=session.get(
                        "deleted_categories", {}
                    ).get(repository, {}),
                    modified_cat_dict=session.get(
                        "modified_categories", {}
                    ).get(repository, {}),
                    message=args["commit_message"]
                )
                session["deleted_categories"]["local"] = {}
                session["modified_categories"]["local"] = {}
                cache.clear()
                return 204
        else:
            # is the edition occuring on an already modified category?
            if (cat_id in session.get(
                    "modified_categories", {}
            ).get(repository, {}).keys()):
                cat_graph = Graph()
                cat_graph.parse(
                    source=StringIO(
                        session["modified_categories"][repository][cat_id]
                    ),
                    format="turtle"
                )
                cat = Category(graph=cat_graph)
            else:
                cat = cat_manager_instance.load_category(cat_id)

            cat.edit_category(new_description=cat_data["description"],
                              new_label=cat_data["label"],
                              new_other_properties=cat_data["properties"])

            session["modified_categories"][repository][cat.cat_id] = str(
                cat.cat_graph.serialize(format="turtle"), "utf-8"
            )

            # Now we must clean the deleted categories list in case the
            # modified category was deleted before being edited
            session["deleted_categories"][repository] = {
                cat_name: cat_content for cat_name, cat_content
                in session["deleted_categories"][repository].items()
                if cat_name != cat.cat_id
            }
            logger.debug("put id: "+cat.cat_id)
        return 204

    def post(self, repository, cat_data):
        """
            API to create a new category

            Args are:
            * repository: the repository where the category cat_id is stored
            * cat_data : the new data of the category, dict of the form:
            {
                "label": edited_cat_label,
                "description": edited_cat_description,
                "properties": [(predicate1, object1), (predicate2, object2)]
            }
            List of predicate is available in config.py, key PROPERTY_LIST
        """
        cat = Category(
            label=cat_data["label"],
            description=cat_data["description"],
            other_properties=cat_data["properties"]
        )

        if cat.cat_id not in session["modified_categories"][repository].keys():
            session["modified_categories"][repository][cat.cat_id] = str(
                cat.cat_graph.serialize(format="turtle"), "utf-8"
            )

        logger.debug("post id: "+cat.cat_id)
        return cat.cat_id, \
               cat.cat_graph.serialize(format='turtle').decode("utf-8"), \
               201

    def delete(self, repository, deleted_cat_id):
        """
            API to delete the category of id cat_id or restore it from the
            deletion list
        """
        # if cat_id is already in deleted categories, we restore it
        if deleted_cat_id in session["deleted_categories"][repository].keys():
            session["deleted_categories"][repository] = {
                cat_name: cat_content for cat_name, cat_content
                in session["deleted_categories"][repository].items()
                if cat_name != deleted_cat_id
            }
            # warning, not safe if 2 files share the same name (or category id)
            # but that shouldn't happen
        else:
            session["deleted_categories"] \
                   [repository] \
                   [deleted_cat_id] = "deleted"
            # Maybe we can store something more useful in there, such as
            # category content when deleted to restore last edit?
            # It would be nice to have symetry between modified and
            # deleted dicts

            # now we must clean the modified categories list in case the
            # deleted category was modified before
            session["modified_categories"][repository] = {
                cat_name: cat_content for cat_name, cat_content
                in session["modified_categories"][repository].items()
                if cat_name != deleted_cat_id
            }
            logger.debug(str(
                {
                    cat_name: cat_content for cat_name, cat_content
                    in session["modified_categories"][repository].items()
                    if cat_name != deleted_cat_id
                }
            ))
            # Now we also have to clean up categories that reference the
            # deleted category
            cat_manager_instance = CategoryManager(
                getattr(
                    catedit.persistence,
                    app.config["PERSISTENCE_CONFIG"]["METHOD"]
                )(repository=repository),
            )
            cat_list = cat_manager_instance.list_categories()
            # first we edit what was already modified before the deletion
            logger.debug(
                session.get("modified_categories", {}).get(repository, {})
            )
            if (
                    deleted_cat_id in
                    session["deleted_categories"][repository].keys()
            ):
                for element in session.get("modified_categories", {}) \
                                      .get(repository, {}).keys():
                    logger.debug(str(element))
                    modified_cat_graph = Graph()
                    modified_cat_graph.parse(
                        source=StringIO(
                            session["modified_categories"][repository][element]
                        ),
                        format="turtle"
                    )
                    modified_cat = Category(graph=modified_cat_graph)
                    if (modified_cat.cat_id !=
                            app.config["CATEGORY_NAMESPACE"] + deleted_cat_id):
                        new_property_list = []
                        for (predicate, obj) in modified_cat.properties:
                            if not (
                                    app.config["PROPERTY_LIST"]
                                    [predicate]
                                    ["object_type"]
                                    == "uriref-category"
                                    and (
                                        obj == (
                                            app.config["CATEGORY_NAMESPACE"]
                                            + deleted_cat_id
                                        )
                                    )
                            ):
                                new_property_list.append((predicate, obj))

                        if new_property_list != modified_cat.properties:
                            logger.debug("Modifying modified category")
                            modified_cat.edit_category(
                                new_other_properties=new_property_list
                            )
                            session["modified_categories"] \
                                   [repository] \
                                   [modified_cat.cat_id] = str(
                                modified_cat.cat_graph
                                            .serialize(format="turtle"),
                                "utf-8"
                            )

                # now we check if an unmodified category reference the deleted
                # category
                for cat in cat_list:
                    if (cat.cat_id not in (
                            list(session["modified_categories"]
                                        [repository].keys())
                            + list(session["deleted_categories"]
                                          [repository].keys())
                    )):
                        new_property_list = []
                        for (predicate, obj) in cat.properties:
                            if not (
                                    app.config["PROPERTY_LIST"]
                                              [predicate]
                                              ["object_type"]
                                    == "uriref-category"
                                    and (
                                        obj ==
                                        app.config["CATEGORY_NAMESPACE"]
                                        + deleted_cat_id
                                    )
                            ):
                                new_property_list.append((predicate, obj))

                        if new_property_list != cat.properties:
                            logger.debug("Modifying untouched category")
                            cat.edit_category(
                                new_other_properties=new_property_list
                            )
                            session["modified_categories"] \
                                   [repository] \
                                   [cat.cat_id] = str(
                                cat.cat_graph.serialize(format="turtle"),
                                "utf-8"
                            )

        logger.debug("delete id: " + deleted_cat_id)
        return 204


class CategoryChangesAPI(Resource):
    """
        API for getting and deleting category changes, returns a dict when
        succesful if category is a modified one, returns only the cat_id if it
        is a deleted one

        All changes and deletions are saved in
        session["modified_categories"][repository] and
        session["deleted_categories"][repository] under the format:
        {changed_cat_id: content_of_changed_cat}
    """
    def get(self, repository, modified_cat_id=None):
        """
            API to get the pending changes for category cat_id
        """

        if modified_cat_id is None:
            return {
                "modified_categories": session.get("modified_categories", {})
                                              .get(repository, {}),
                "deleted_categories": session.get("deleted_categories", {})
                                             .get(repository, {})
            }, 200
        else:
            if modified_cat_id in session.get("modified_categories", {}) \
                                         .get(repository, {}):
                return {
                    "type": "modified",
                    "category": {
                        modified_cat_id: session["modified_categories"]
                                                [repository]
                                                [modified_cat_id]
                    }
                }, 200
            if modified_cat_id in session.get("deleted_categories", {}) \
                                         .get(repository, {}):
                return {
                    "type": "deleted",
                    "category": {
                        modified_cat_id: session["deleted_categories"]
                                                [repository]
                                                [modified_cat_id]
                    }
                }, 200
            return {
                "type": "untouched",
                "category": modified_cat_id
            }, 200

    def delete(self, repository, modified_cat_id=None):
        """
            API to delete the category cat_id from the changelist or if cat_id
            is None, delete the whole changelist
        """
        if modified_cat_id is not None:
            session["modified_categories"][repository] = {
                cat_name: cat_content for cat_name, cat_content
                in session["modified_categories"][repository].items()
                if cat_name != modified_cat_id
            }
            session["deleted_categories"][repository] = {
                cat_name: cat_content for cat_name, cat_content
                in session["deleted_categories"][repository].items()
                if cat_name != modified_cat_id
            }
        else:
            session["modified_categories"][repository] = {}
            session["deleted_categories"][repository] = {}
        return 204
