"""
social.py:
The views functions that handle comment display and posting
"""

from catedit import app
from catedit.views.utils import check_user_status_and_repo_access, \
                                get_comments, post_comment, get_commits, \
                                get_issues, get_category_list_for_commit, \
                                Pagination
from flask import render_template, request, redirect, url_for, \
                  abort, Blueprint, session
from flask_wtf import Form
from wtforms import TextAreaField, StringField
from wtforms.validators import DataRequired

module = Blueprint('social', __name__)
logger = app.logger


@module.route("/<string:repository>/social",
              methods=["GET"])
def index(repository):
    """
        View that displays every changeset and thread in general discussion
        and links to every thread
    """
    if repository not in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]:
        abort(404)

    check_user_status_and_repo_access(repository)

    changeset_list = get_commits(repository)
    discussion_list = get_issues(repository)

    return render_template(
        "social/index.html",
        current_repository=repository,
        discussion_list=discussion_list,
        changeset_list=changeset_list
    )


class CommentForm(Form):
    """
        Custom form class for commiting changes
    """
    comment_field = TextAreaField(
        "Poster un commentaire: ",
        validators=[DataRequired()]
    )


@module.route("/<string:repository>/changesets/<string:changeset_id>",
              methods=["GET", "POST"],
              defaults={"per_page": 10, "page": 1})
@module.route(
    "/<string:repository>/changesets/<string:changeset_id>"
    + "/page/<int:page>-per_page-<int:per_page>",
    methods=["GET", "POST"]
)
def changeset(repository, changeset_id, per_page, page):
    """
        View that displays a snapshot of the repository as it was for a given
        changeset, and the related discussion to this changeset. Allows
        users to comment on it.

        For re-building the old category set we use the tree and blob data:
        1) first we get the commit of id changeset_id
        2) we get the tree associated with this commit
        3) we get all the blobs referenced by this tree
        4) we build categories from the blobs contents
        --> Will probably be incorporated into a function later on

    """
    if repository not in app.config["PERSISTENCE_CONFIG"]["REPOSITORY_LIST"]:
        abort(404)

    check_user_status_and_repo_access(repository)
    comment_form = CommentForm()
    comments_list = get_comments(
        repository=repository,
        thread_type="commits",
        thread_id=changeset_id,
        page=page,
        per_page=per_page
    )
    pagination=None
    if session.get("pagination_links", None) is not None:
        # If there are multiple pages we create a pagination class that
        # will be sent to the template
        pagination = Pagination(
            page=session["pagination_links"]["current_page"],
            per_page=per_page,
            last_page=session["pagination_links"]["last_page"]
        )
    cat_list = get_category_list_for_commit(repository, changeset_id)

    if request.method == "GET":
        return render_template(
            "social/changeset.html",
            cat_list=cat_list,
            comments=comments_list,
            changeset_id=changeset_id,
            comment_form=comment_form,
            current_repository=repository,
            pagination=pagination
        )
    elif request.method == "POST" and comment_form.validate_on_submit():
        return_id = post_comment(
            repository=repository,
            thread_type="commits",
            thread_id=changeset_id,
            comment_body=comment_form.comment_field.data
        )
        return redirect(url_for(
            "social.changeset",
            repository=repository,
            changeset_id=return_id,
            per_page=per_page
        ))
    else:
        # Form didn't validate
        return render_template(
            "social/changeset.html",
            cat_list=cat_list,
            comments=comments_list,
            changeset_id=changeset_id,
            comment_form=comment_form,
            current_repository=repository,
            pagination=pagination
        )


class NewDiscussionForm(Form):
    """
        Custom form class for commiting changes
    """
    discussion_title = StringField(
        "Titre de la discussion: ",
        validators=[DataRequired()]
    )
    comment_field = TextAreaField(
        "Commentaire d'ouverture (facultatif): "
    )


@module.route(
    "/<string:repository>/discussions/<string:discussion_id>",
    methods=["GET", "POST"],
    defaults={"per_page": 10, "page": 1}
)
@module.route(
    "/<string:repository>/discussions/<string:discussion_id>"
    + "/page/<int:page>-per_page-<int:per_page>",
    methods=["GET", "POST"]
)
def discussion(repository, discussion_id, per_page, page):
    """
        View that displays the discussion of a given id and allows users
        to post comments on it.
    """
    check_user_status_and_repo_access(repository)

    comment_form = None
    pagination = None
    if discussion_id == "new":
        comment_form = NewDiscussionForm()
        comments_list = []
    else:
        comment_form = CommentForm()
        comments_list = get_comments(
            repository=repository,
            thread_type="issues",
            thread_id=discussion_id,
            page=page,
            per_page=per_page
        )
        if session.get("pagination_links", None) is not None:
            # If there are multiple pages we create a pagination class that
            # will be sent to the template
            pagination = Pagination(
                page=session["pagination_links"]["current_page"],
                per_page=per_page,
                last_page=session["pagination_links"]["last_page"]
            )

    if request.method == "GET":
        return render_template(
            "social/discussion.html",
            comments=comments_list,
            comment_form=comment_form,
            current_repository=repository,
            discussion_id=discussion_id,
            pagination=pagination
        )
    elif request.method == "POST" and comment_form.validate_on_submit():
            if discussion_id == "new":
                return_id = post_comment(
                    repository=repository,
                    thread_type="issues",
                    thread_id=discussion_id,
                    comment_body=comment_form.comment_field.data,
                    thread_title=comment_form.discussion_title.data
                )
            else:
                return_id = post_comment(
                    repository=repository,
                    thread_type="issues",
                    thread_id=discussion_id,
                    comment_body=comment_form.comment_field.data
                )
            return redirect(url_for(
                "social.discussion",
                repository=repository,
                discussion_id=return_id,
                page=session["pagination_links"]["last_page"],
                per_page=per_page
            ))
    else:
        # Form didn't validate
        return render_template(
            "social/discussion.html",
            comments=comments_list,
            comment_form=comment_form,
            current_repository=repository,
            discussion_id=discussion_id,
            pagination=pagination
        )
