import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';

import { FormattedMessage } from 'react-intl';
import {
  Tab,
  Tabs,
  TabList,
  TabPanel,
} from 'react-tabs';

import AnnotationsCards from '../ui/AnnotationsCards';
import DefinitionsCards from '../ui/DefinitionsCards';
import IssoThread from '../ui/IssoThread';
import { getTermId } from '../utils';

import './term.scss';

const TermPage = ({
  term,
  term64,
  defAndRef: defAndRefProp,
  messageId,
  activeTab,
  location,
  metacategories,
  topics,
  viaBaseUrl,
  discussionUrl,
}) => {
  const defAndRef = defAndRefProp || { definitions: [], references: [] };
  let { pathname } = location;
  const m = pathname.match(/(\/[^/]+\/[^/]+)(?:\/\d)?(?:\/.+)?/);
  pathname = m ? m[1] : pathname;
  return (
    <div className="term-entry">
      <h2>{term}</h2>
      <Tabs selectedTabPanelClassName="show active" selectedTabClassName="active" defaultIndex={activeTab || 0} forceRenderTabPanel>
        <TabList className="nav nav-tabs">
          <Tab className="nav-item nav-link">
            <FormattedMessage id="pages.glossary.definitions" defaultMessage="Definitions" />
            {' ('}
            {defAndRef.definitions && defAndRef.definitions.length}
            {')'}
          </Tab>
          <Tab className="nav-item nav-link">
            <FormattedMessage id="pages.glossary.references" defaultMessage="References" />
            {' ('}
            {defAndRef.references && defAndRef.references.length}
            {')'}
          </Tab>
          <Tab className="nav-item nav-link">
            <FormattedMessage id="pages.glossary.discussions" defaultMessage="Discussions" />
            {' ('}
            {defAndRef.messagesCount}
            {')'}
          </Tab>
        </TabList>
        <div className="tab-content term-tab-content">
          <TabPanel className="tab-pane fade">
            <DefinitionsCards annotations={defAndRef.definitions} metacategories={metacategories}><h4 className="text-muted text-capitalize"><FormattedMessage id="page.term.noDefinition" defaultMessage="No definition yet" /></h4></DefinitionsCards>
          </TabPanel>
          <TabPanel className="tab-pane fade">
            <AnnotationsCards
              annotations={defAndRef.references}
              viaBaseUrl={viaBaseUrl}
              metacategories={metacategories}
              categories={topics}
            >
              <h4 className="text-muted text-capitalize">
                <FormattedMessage id="page.term.noReference" defaultMessage="No reference yet" />
              </h4>
            </AnnotationsCards>
          </TabPanel>
          <TabPanel className="tab-pane fade">
            <IssoThread
              issoId={term64}
              pathname={`${pathname}/2`}
              messageId={messageId}
              discussionUrl={discussionUrl}
            />
          </TabPanel>
        </div>
      </Tabs>
    </div>
  );
};

TermPage.propTypes = {
  location: PropTypes.object.isRequired,
  term: PropTypes.string.isRequired,
  term64: PropTypes.string.isRequired,
  viaBaseUrl: PropTypes.string.isRequired,
  discussionUrl: PropTypes.string.isRequired,
  defAndRef: PropTypes.object,
  messageId: PropTypes.string,
  activeTab: PropTypes.number,
  metacategories: PropTypes.arrayOf(PropTypes.object).isRequired,
  topics: PropTypes.arrayOf(PropTypes.string).isRequired,
  dashboardId: PropTypes.string, // eslint-disable-line react/no-unused-prop-types
};

TermPage.defaultProps = {
  defAndRef: {},
  messageId: null,
  activeTab: 0,
  dashboardId: '',
};

export default connect(
  (state, props) => {
    const { match, dashboardId } = props;
    const { params } = match;

    let { term } = (params || {});
    term = term ? decodeURIComponent(term) : term;

    const defAndRef = term ? state.terms[term] : {};

    const term64 = getTermId(term, dashboardId);

    const { messageId } = (params || {});

    const activeTab = Number((match.params || {}).activeTab || 0);

    return {
      term,
      term64,
      defAndRef,
      messageId,
      activeTab,
    };
  },
)(TermPage);
